package nemosofts.notes.app.utils;

import android.app.AlarmManager;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;

import nemosofts.notes.app.receiver.ReminderReceiver;
import nemosofts.notes.app.item.Note;

public class ReminderScheduler {

    public static final String CHANNEL_ID = "note_reminders";
    private static final String CHANNEL_NAME = "Note reminders";

    private ReminderScheduler() {
        // Utility class
    }

    public static void scheduleReminder(@NonNull Context context, @NonNull Note note) {
        if (note.getId() <= 0 || !note.isReminderEnabled()) {
            cancelReminder(context, note.getId());
            return;
        }
        long triggerAtMillis = note.getReminderTime();
        if (triggerAtMillis <= System.currentTimeMillis()) {
            cancelReminder(context, note.getId());
            return;
        }

        createNotificationChannel(context);

        AlarmManager alarmManager = ContextCompat.getSystemService(context, AlarmManager.class);
        if (alarmManager == null) {
            return;
        }
        PendingIntent pendingIntent = buildPendingIntent(context, note.getId());
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S && !alarmManager.canScheduleExactAlarms()) {
                // Fallback to AlarmClock which is user visible and allowed without the exact alarm exemption.
                AlarmManager.AlarmClockInfo info = new AlarmManager.AlarmClockInfo(triggerAtMillis, pendingIntent);
                alarmManager.setAlarmClock(info, pendingIntent);
            } else {
                alarmManager.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerAtMillis, pendingIntent);
            }
        } catch (SecurityException ignored) {
            // Fallback to inexact alarm to avoid crashes when exact scheduling is blocked.
            alarmManager.setAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerAtMillis, pendingIntent);
        }
    }

    public static void cancelReminder(@NonNull Context context, int noteId) {
        if (noteId <= 0) {
            return;
        }
        AlarmManager alarmManager = ContextCompat.getSystemService(context, AlarmManager.class);
        if (alarmManager == null) {
            return;
        }
        PendingIntent pendingIntent = buildPendingIntent(context, noteId);
        alarmManager.cancel(pendingIntent);
    }

    public static void createNotificationChannel(@NonNull Context context) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            return;
        }
        NotificationManager manager = ContextCompat.getSystemService(context, NotificationManager.class);
        if (manager == null) {
            return;
        }
        NotificationChannel existing = manager.getNotificationChannel(CHANNEL_ID);
        if (existing != null) {
            return;
        }
        NotificationChannel channel = new NotificationChannel(CHANNEL_ID, CHANNEL_NAME, NotificationManager.IMPORTANCE_HIGH);
        channel.setDescription("Reminders for your notes.");
        manager.createNotificationChannel(channel);
    }

    private static PendingIntent buildPendingIntent(Context context, int noteId) {
        Intent intent = new Intent(context, ReminderReceiver.class);
        intent.setAction(ReminderReceiver.ACTION_NOTIFY);
        intent.putExtra(ReminderReceiver.EXTRA_NOTE_ID, noteId);
        int flags = PendingIntent.FLAG_UPDATE_CURRENT;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            flags |= PendingIntent.FLAG_IMMUTABLE;
        }
        return PendingIntent.getBroadcast(context, noteId, intent, flags);
    }
}
